/*
 * Program: x48 with GTK support
 * Version: 0.5.0 (GTK)
 * File: init.c
 * Description: init emulator, load rom, saturn state, ram and ports
 * History:
 *          Ver              Modified                Date
 *          ===  ================================  ========
 */


#include <config.h>
#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <string.h>
#include <pwd.h>
#include <unistd.h>
#include <sys/stat.h>
#include <sys/types.h>
#ifdef SUNOS
#include <memory.h>
#endif

#include "global.h"
#include "hp48.h"


unsigned int rom_size = 0;
long         ram_size;
long         port1_size, port2_size;
long         port1_mask, port2_mask;
short        port1_is_ram, port2_is_ram;


/* Local functions */
int  read_files();
int  read_rom_file( char *fname, unsigned char **mem, int *size );
int  read_saturn_file( char *fname );
int  read_mem_file( char *name, word_4 *mem, int size );
void saturn_config_init();
void init_saturn();
int  read_8( FILE *fp, word_8 *var );
int  read_char( FILE *fp, char *var );
int  read_16( FILE *fp, word_16 *var );
int  read_32( FILE *fp, word_32 *var );
int  read_u_long( FILE *fp, unsigned long *var );

int  write_files();
int  write_saturn_file( char *fname );
int  write_mem_file( char *name, word_4 *mem, int size );
int  write_8( FILE *fp, word_8 *var );
int  write_char( FILE *fp, char *var );
int  write_16( FILE *fp, word_16 *var );
int  write_32( FILE *fp, word_32 *var );
int  write_u_long( FILE *fp, unsigned long *var );


/*****************
 * init_emulator *
 *****************/
int init_emulator()
{
   if( read_files() )
      return -1;
   if( opt_reset )
      saturn.PC = 0x01FC6; /* SoftStart */

   init_display();
   init_serial();

   return 0;
}


/**************************
 * read_files
 **************************/
int read_files()
{
   char         path[1024], tmp[1024];
   struct stat  st;


   /* saturn */
   if( opt_initialize )
   {
      if( opt_verbose )
         printf( "%s:\tInitializing new saturn structure.\n", x48name );
      init_saturn();
   }
   else
   {
      sprintf( path, "%s%s/%s", homedir, X48HOMEDIR,
               opt_gx ? SATURN_GX_FILE : SATURN_SX_FILE );
      if( read_saturn_file(path) )
      {
         sprintf( tmp, "Can't read <%s> saturn file. Try '%s -initialize'.\n",
                  path, x48name );
         error( tmp);
      }
      dev_memory_init();
      saturn_config_init();
   }

   /* rom */
   saturn.rom = (word_4 *)NULL;
   sprintf( path, "%s%s", x48dir, opt_gx ? ROM_GX_FILE : ROM_SX_FILE );
   if( read_rom_file(path, &saturn.rom, &rom_size) )
   {
      sprintf( tmp, "Can't read <%s> rom file.\n", path );
      error( tmp );
   }

   /* ram = port0 */
   ram_size = opt_gx ? RAM_SIZE_GX : RAM_SIZE_SX;
   saturn.ram = (word_4 *) NULL;
   if( !(saturn.ram = (word_4 *) malloc(ram_size)) )
      error( "Can't allocate enough memory for ram.\n" );
   sprintf( path, "%s%s/%s", homedir, X48HOMEDIR,
            opt_gx ? PORT0_GX_FILE : PORT0_SX_FILE );
   if( read_mem_file(path, saturn.ram, ram_size) )
   {
      printf( "%s:\tCan't read <%s> ram file.\n", x48name, path );
      memset( saturn.ram, 0, ram_size );
      force_save = 1;
   }

   /* port1 */
   saturn.card_status = 0;
   port1_size = 0;
   port1_mask = 0;
   port1_is_ram = 0;
   saturn.port1 = (unsigned char *) 0;
   sprintf( path, "%s%s/%s", homedir, X48HOMEDIR,
            opt_gx ? PORT1_GX_FILE : PORT1_SX_FILE );
   if( stat(path, &st) >= 0 )
   {
      port1_size = 2 * st.st_size;
      if( (port1_size == 0x10000) || (port1_size == 0x40000) )
      {
         if( ! (saturn.port1 = (word_4 *)malloc(port1_size)) )
         {
            sprintf( tmp, "Can't allocate memory for PORT1[%ld].\n",
                     port1_size );
            error( tmp );
         }
         else if( read_mem_file(path, saturn.port1, port1_size) )
         {
            port1_size = 0;
            port1_is_ram = 0;
         }
         else
         {
            port1_is_ram = (st.st_mode & S_IWUSR) ? 1 : 0;
            port1_mask = port1_size - 1;
         }
      }
   }
   else
      if( opt_verbose )
         printf( "%s:\t<%s> not found.\n", x48name, path );
   if( opt_gx )
   {
      saturn.card_status |= (port1_size > 0) ? 2 : 0;
      saturn.card_status |= port1_is_ram ? 8 : 0;
   }
   else
   {
      saturn.card_status |= (port1_size > 0) ? 1 : 0;
      saturn.card_status |= port1_is_ram ? 4 : 0;
   }

   /* port2 */
   port2_size = 0;
   port2_mask = 0;
   port2_is_ram = 0;
   saturn.port2 = (unsigned char *) 0;
   sprintf( path, "%s%s/%s", homedir, X48HOMEDIR,
            opt_gx ? PORT2_GX_FILE : PORT2_SX_FILE );
   if( stat(path, &st) >= 0 )
   {
      port2_size = 2 * st.st_size;
      if( (opt_gx && ((port2_size % 0x40000) == 0)) ||
          (!opt_gx && ((port2_size == 0x10000) || (port2_size == 0x40000))) )
      {
         if( ! (saturn.port2 = (word_4 *)malloc(port2_size)) )
         {
            sprintf( tmp, "Can't allocate memory for PORT2[%ld].\n",
                     port2_size );
            error( tmp );
         }
         else if( read_mem_file(path, saturn.port2, port2_size) )
         {
            port2_size = 0;
            port2_is_ram = 0;
         }
         else
         {
            port2_is_ram = (st.st_mode & S_IWUSR) ? 1 : 0;
            port2_mask = port2_size - 1;
         }
      }
   }
   else
      if( opt_verbose )
         printf( "%s:\t<%s> not found.\n", x48name, path );
   if( opt_gx )
   {
      saturn.card_status |= (port2_size > 0) ? 1 : 0;
      saturn.card_status |= port2_is_ram ? 4 : 0;
   }
   else
   {
      saturn.card_status |= (port2_size > 0) ? 2 : 0;
      saturn.card_status |= port2_is_ram ? 8 : 0;
   }

   return 0;
}


/*****************
 * Read rom file *
 *****************/
int read_rom_file( char *fname, unsigned char **mem, int *size )
{
   FILE          *from;
   struct stat   st;
   unsigned char four[4];
   word_8        byte;
   int           i, j;
   char          tmp[1024];


   *mem = NULL;
   *size = 0;

   if( ! (from = fopen(fname, "r")) )
   {
      sprintf( tmp, "Can't open <%s> rom file.\n", fname);
      error( tmp );
   }
   if( stat(fname, &st) < 0 )
   {
      sprintf( tmp, "Can't stat <%s> rom file.\n", fname);
      fclose( from );
      error( tmp );
   }
   if( fread(four, 1, 4, from) != 4 )
   {
      sprintf( tmp, "Can't read first 4 bytes of <%s>.\n", fname);
      fclose( from );
      error( tmp );
   }

   if( four[0] == 0x02 && four[1] == 0x03 &&
       four[2] == 0x06 && four[3] == 0x09 )
   {
      *size = st.st_size;
   }
   else if( four[0] == 0x32 && four[1] == 0x96 &&
            four[2] == 0x1b && four[3] == 0x80 )
   {
      *size = st.st_size * 2;
   }
   else
   {
      sprintf( tmp, "<%s> is not a valid HP48 ROM.\n", fname );
      fclose( from );
      error( tmp );
    }

    if( *size != (opt_gx ? ROM_SIZE_GX : ROM_SIZE_SX) )
    {
       sprintf( tmp, "<%s> is not a valid HP48 ROM for a %s.\n", fname,
                opt_gx ? "GX" : "SX" );
       fclose( from );
       error( tmp );
    }

    fseek( from, 0, SEEK_SET );
    if( ! (*mem = (unsigned char *) malloc(*size)) )
    {
       fclose( from );
       error( "Can't allocate enough memory for ram.\n" );
    }

    if( *size == st.st_size )
    {
       if( fread(*mem, 1, (size_t)*size, from) != *size )
       {
          free(*mem);
          fclose( from );
          sprintf( tmp, "Can't read <%s> rom file entirely.\n", fname);
          error( tmp );
       }
    }
    else
    {
       for( i = 0, j = 0; i < *size; i++ )
       {
          if( (byte = fread(&byte, 1, 1, from)) != 1)
             break;
          (*mem)[j++] = byte & 0x0F;
          (*mem)[j++] = (byte >> 4) & 0x0F;
       }
       if( i != *size )
       {
          free(*mem);
          fclose( from );
          sprintf( tmp, "Can't read <%s> rom file entirely.\n", fname);
          error( tmp );
       }
    }

    fclose( from );

   if( opt_verbose )
      printf( "%s:\tReading <%s> rom file.\n", x48name, fname );
    return 0;
}


/********************
 * Read saturn file *
 ********************/
int read_saturn_file( char *fname )
{
   FILE          *fp;
   int           i;
   unsigned long v1, v2;


   if( ! (fp = fopen(fname, "r")) )
      return -1;

   /* ok, file is open, try to read the MAGIC number */
   read_u_long( fp, &saturn.magic );
   if( saturn.magic != X48_MAGIC )
   {
      printf( "%s:\t<%s> is not a valid saturn file.\n", x48name, fname );
      return -1;
   }

   /* read version */
   for( i = 0; i < 4; i++ )
      if( !read_char(fp, &saturn.version[i]) )
         return -1;
   v1 = ((int)saturn.version[0] & 0xff) << 24;
   v1 |= ((int)saturn.version[1] & 0xff) << 16;
   v1 |= ((int)saturn.version[2] & 0xff) << 8;
   v1 |= ((int)saturn.version[3] & 0xff);
   v2 = ((int)VERSION_MAJOR & 0xff) << 24;
   v2 |= ((int)VERSION_MINOR & 0xff) << 16;
   v2 |= ((int)PATCHLEVEL & 0xff) << 8;
   v2 |= ((int)COMPILE_VERSION & 0xff);

   if( (v1 & 0xffffff00) < (v2 & 0xffffff00) )
   {
      printf( "%s:\t<%s> is not a version %d.%d.%d file, converting.\n",
              x48name, fname,
              saturn.version[0], saturn.version[1], saturn.version[2] );
   }
   else if( (v2 & 0xffffff00) < (v1 & 0xffffff00) )
   {
      printf( "%s:\t<%s> is a version %d.%d.%d file, trying ...\n",
              x48name, fname,
              saturn.version[0], saturn.version[1], saturn.version[2] );
   }

   /* version 0.4.x or 0.5.x, read in the saturn_t struct */
   for( i = 0; i < 16; i++ )
      if( !read_8(fp, &saturn.A[i]) )
         return -1;
   for( i = 0; i < 16; i++ )
      if( !read_8(fp, &saturn.B[i]) )
         return -1;
   for( i = 0; i < 16; i++ )
      if( !read_8(fp, &saturn.C[i]) )
         return -1;
   for( i = 0; i < 16; i++ )
      if( !read_8(fp, &saturn.D[i]) )
         return -1;
   if( !read_32(fp, &saturn.d[0]) )
      return -1;
   if( !read_32(fp, &saturn.d[1]) )
      return -1;
   if( !read_8(fp, &saturn.P) )
      return -1;
   if( !read_32(fp, &saturn.PC) )
      return -1;
   for( i = 0; i < 16; i++ )
      if( !read_8(fp, &saturn.R0[i]) )
         return -1;
   for( i = 0; i < 16; i++ )
      if( !read_8(fp, &saturn.R1[i]) )
         return -1;
   for( i = 0; i < 16; i++ )
      if( !read_8(fp, &saturn.R2[i]) )
         return -1;
   for( i = 0; i < 16; i++ )
      if( !read_8(fp, &saturn.R3[i]) )
         return -1;
   for( i = 0; i < 16; i++ )
      if( !read_8(fp, &saturn.R4[i]) )
         return -1;
   for( i = 0; i < 4; i++ )
      if( !read_8(fp, &saturn.IN[i]) )
         return -1;
   for( i = 0; i < 3; i++ )
      if( !read_8(fp, &saturn.OUT[i]) )
         return -1;
   if( !read_8(fp, &saturn.CARRY) )
      return -1;
   for( i = 0; i < NR_PSTAT; i++ )
      if( !read_8(fp, &saturn.PSTAT[i]) )
         return -1;
   if( !read_8(fp, &saturn.XM) )
      return -1;
   if( !read_8(fp, &saturn.SB) )
      return -1;
   if( !read_8(fp, &saturn.SR) )
      return -1;
   if( !read_8(fp, &saturn.MP) )
      return -1;
   if( !read_8(fp, &saturn.hexmode) )
      return -1;
   for( i = 0; i < NR_RSTK; i++ )
      if( !read_32(fp, &saturn.rstk[i]) )
         return -1;
   if( !read_16(fp, (word_16 *)&saturn.rstkp) )
      return -1;
   for( i = 0; i < 9; i++ )
      if( !read_16(fp, (word_16 *)&saturn.keybuf.rows[i]) )
         return -1;
   if( !read_8(fp, &saturn.intenable) )
      return -1;
   if( !read_8(fp, &saturn.int_pending) )
      return -1;
   if( !read_8(fp, &saturn.kbd_ien) )
      return -1;
   if( !read_8(fp, &saturn.disp_io) )
      return -1;
   if( !read_8(fp, &saturn.contrast_ctrl) )
      return -1;
   if( !read_8(fp, &saturn.disp_test) )
      return -1;
   if( !read_16(fp, &saturn.crc) )
      return -1;
   if( !read_8(fp, &saturn.power_status) )
      return -1;
   if( !read_8(fp, &saturn.power_ctrl) )
      return -1;
   if( !read_8(fp, &saturn.mode) )
      return -1;
   if( !read_8(fp, &saturn.annunc) )
      return -1;
   if( !read_8(fp, &saturn.baud) )
      return -1;
   if( !read_8(fp, &saturn.card_ctrl) )
      return -1;
   if( !read_8(fp, &saturn.card_status) )
      return -1;
   if( !read_8(fp, &saturn.io_ctrl) )
      return -1;
   if( !read_8(fp, &saturn.rcs) )
      return -1;
   if( !read_8(fp, &saturn.tcs) )
      return -1;
   if( !read_8(fp, &saturn.rbr) )
      return -1;
   if( !read_8(fp, &saturn.tbr) )
      return -1;
   if( !read_8(fp, &saturn.sreq) )
      return -1;
   if( !read_8(fp, &saturn.ir_ctrl) )
      return -1;
   if( !read_8(fp, &saturn.base_off) )
      return -1;
   if( !read_8(fp, &saturn.lcr) )
      return -1;
   if( !read_8(fp, &saturn.lbr) )
      return -1;
   if( !read_8(fp, &saturn.scratch) )
      return -1;
   if( !read_8(fp, &saturn.base_nibble) )
      return -1;
   if( !read_32(fp, &saturn.disp_addr) )
      return -1;
   if( !read_16(fp, &saturn.line_offset) )
      return -1;
   if( !read_8(fp, &saturn.line_count) )
      return -1;
   if( !read_16(fp, &saturn.unknown) )
      return -1;
   if( !read_8(fp, &saturn.t1_ctrl) )
      return -1;
   if( !read_8(fp, &saturn.t2_ctrl) )
      return -1;
   if( !read_32(fp, &saturn.menu_addr) )
      return -1;
   if( !read_8(fp, &saturn.unknown2) )
      return -1;
   if( !read_char(fp, &saturn.timer1) )
      return -1;
   if( !read_32(fp, &saturn.timer2) )
      return -1;
   if( !read_32(fp, &saturn.t1_instr) )
      return -1;
   if( !read_32(fp, &saturn.t2_instr) )
      return -1;
   if( !read_16(fp, (word_16 *)&saturn.t1_tick) )
      return -1;
   if( !read_16(fp, (word_16 *)&saturn.t2_tick) )
      return -1;
   if( !read_32(fp, &saturn.i_per_s) )
      return -1;
   if( !read_16(fp, &saturn.bank_switch) )
      return -1;
   for( i = 0; i < NR_MCTL; i++ )
   {
      if( !read_16(fp, &saturn.mem_cntl[i].unconfigured) )
         return -1;
      if( !read_32(fp, &saturn.mem_cntl[i].config[0]) )
         return -1;
      if( !read_32(fp, &saturn.mem_cntl[i].config[1]) )
         return -1;
   }

   fclose( fp );

   if( opt_verbose )
      printf( "%s:\tReading <%s> saturn file.\n", x48name, fname );
   return 0;
}


/*****************
 * read_mem_file *
 *****************/
int read_mem_file( char *name, word_4 *mem, int size )
{
   FILE        *fp;
   struct stat st;
   word_8      byte;
   int         i, j;

   if( !(fp = fopen(name, "r")) )
      return -1;
   if( stat(name, &st) < 0 )
   {
      fclose( fp );
      return -1;
   }
   if( st.st_size == size )
   {
      /* size is same as memory size, old version file */
      if( fread(mem, 1, (size_t)size, fp) != size )
      {
         fclose( fp );
         return -1;
      }
   }
   else
   {
      for( i = 0, j = 0; i < size / 2; i++ )
      {
         if( fread(&byte, 1, 1, fp) != 1)
            break;
         mem[j++] = (word_4) ( (int) byte & 0x0F );
         mem[j++] = (word_4) (( (int) byte >> 4) & 0x0F );
      }
      if( i != size / 2 )
      {
         free( mem );
         fclose( fp );
         return -1;
      }
   }

   fclose( fp );

   if( opt_verbose )
      printf( "%s:\tReading <%s> file.\n", x48name, name );
   return 0;
}


/**********************
 * saturn_config_init *
 **********************/
void saturn_config_init()
{
   saturn.version[0] = VERSION_MAJOR;
   saturn.version[1] = VERSION_MINOR;
   saturn.version[2] = PATCHLEVEL;
   saturn.version[3] = COMPILE_VERSION;
   memset( &device, 0, sizeof(device) );
   device.display_touched = 1;
   device.contrast_touched = 1;
   device.baud_touched = 1;
   device.ann_touched = 1;
   saturn.rcs = 0x0;
   saturn.tcs = 0x0;
   saturn.lbr = 0x0;
}


/***************
 * init_saturn *
 ***************/
void init_saturn()
{
   int i;

   memset( &saturn, 0, sizeof(saturn) - 4 * sizeof(unsigned char *) );
   saturn.PC = 0x00000;
   saturn.magic = X48_MAGIC;
   saturn.t1_tick = 8192;
   saturn.t2_tick = 16;
   saturn.i_per_s = 0;
   saturn.version[0] = VERSION_MAJOR;
   saturn.version[1] = VERSION_MINOR;
   saturn.version[2] = PATCHLEVEL;
   saturn.version[3] = COMPILE_VERSION;
   saturn.hexmode = HEX;
   saturn.rstkp = -1;
   saturn.intenable = 1;
   saturn.int_pending = 0;
   saturn.kbd_ien = 1;
   saturn.timer1 = 0;
   saturn.timer2 = 0x2000;
   saturn.bank_switch = 0;
   for( i = 0; i < NR_MCTL; i++ )
   {
      if( i == 0 )
         saturn.mem_cntl[i].unconfigured = 1;
      else if( i == 5 )
         saturn.mem_cntl[i].unconfigured = 0;
      else
         saturn.mem_cntl[i].unconfigured = 2;
      saturn.mem_cntl[i].config[0] = 0;
      saturn.mem_cntl[i].config[1] = 0;
   }
   dev_memory_init();
}


int read_8( FILE *fp, word_8 *var )
{
   unsigned char tmp;

   if( fread(&tmp, 1, 1, fp) != 1 )
   {
      if( opt_verbose )
         fprintf( stderr, "%s:\tCan\'t read word_8.\n", x48name );
      return 0;
   }
   *var = tmp;
   return 1;
}


int read_char( FILE *fp, char *var )
{
   char tmp;

   if( fread(&tmp, 1, 1, fp) != 1 )
   {
      if( opt_verbose )
         fprintf( stderr, "%s:\tCan\'t read char.\n", x48name );
      return 0;
   }
   *var = tmp;
   return 1;
}


int read_16( FILE *fp, word_16 *var )
{
   unsigned char tmp[2];

   if( fread(&tmp[0], 1, 2, fp) != 2 )
   {
      if( opt_verbose )
         fprintf( stderr, "%s:\tCan\'t read word_16.\n", x48name );
      return 0;
   }
   *var = tmp[0] << 8;
   *var |= tmp[1];
   return 1;
}


int read_32( FILE *fp, word_32 *var )
{
   unsigned char tmp[4];

   if( fread(&tmp[0], 1, 4, fp) != 4 )
   {
      if( opt_verbose )
         fprintf( stderr, "%s:\tCan\'t read word_32.\n", x48name );
      return 0;
   }
   *var = tmp[0] << 24;
   *var |= tmp[1] << 16;
   *var |= tmp[2] << 8;
   *var |= tmp[3];
   return 1;
}


int read_u_long( FILE *fp, unsigned long *var )
{
   unsigned char tmp[4];

   if( fread(&tmp[0], 1, 4, fp) != 4 )
   {
      if( opt_verbose )
         fprintf( stderr, "%s:\tCan\'t read unsigned long.\n", x48name );
      return 0;
   }
   *var = tmp[0] << 24;
   *var |= tmp[1] << 16;
   *var |= tmp[2] << 8;
   *var |= tmp[3];
   return 1;
}



/**************************************************************************
 **************************************************************************
 **************************************************************************
 **************************************************************************
 **************************************************************************/


/*****************
 * exit_emulator *
 *****************/
int exit_emulator()
{
   if( opt_save || force_save )
      write_files();
   if( opt_save_cfg || force_save_cfg )
      write_x48rc();
   return 0;
}


/**************************
 * write_files
 **************************/
int write_files()
{
   char        path[1024], fnam[1024];
   struct stat st;
   int         ram_size, make_dir, in_tmp;


   in_tmp = 0;
   make_dir = 0;
   sprintf( path, "%s%s", homedir, X48HOMEDIR );
   if( stat(path, &st) == -1 )
   {
      if( errno == ENOENT )
      {
         make_dir = 1;
      }
      else
      {
         printf( "%s:\tCan\'t stat <%s>, saving to '/tmp'.\n", x48name,
                 path );
         in_tmp = 1;
      }
   }
   else
   {
      if( !S_ISDIR(st.st_mode) )
      {
         printf( "%s:\t<%s> is no directory, saving to '/tmp'\n", x48name,
                 path);
         in_tmp = 1;
      }
   }

   if( make_dir )
   {
      if( mkdir(path, 0777) == -1 )
      {
         printf( "%s:\t Can't mkdir <%s>, saving to '/tmp'.\n", x48name,
                 path);
         in_tmp = 1;
      }
   }


   /* saturn */
   sprintf( fnam, "%s%s/%s", in_tmp ? "/tmp" : homedir,
            in_tmp ? "" : X48HOMEDIR,
            opt_gx ? SATURN_GX_FILE : SATURN_SX_FILE );
   if( write_saturn_file(fnam) )
      printf( "%s:\tCan't write <%s> saturn file.\n", x48name, fnam );

   /* ram = port0 */
   ram_size = opt_gx ? RAM_SIZE_GX : RAM_SIZE_SX;
   sprintf( fnam, "%s%s/%s", in_tmp ? "/tmp" : homedir,
            in_tmp ? "" : X48HOMEDIR,
            opt_gx ? PORT0_GX_FILE : PORT0_SX_FILE );
   if( write_mem_file(fnam, saturn.ram, ram_size) )
      printf( "%s:\tCan't write <%s> ram file.\n", x48name, fnam );

   /* port1 */
   if( (port1_size > 0) && port1_is_ram )
   {
      sprintf( fnam, "%s%s/%s", in_tmp ? "/tmp" : homedir,
               in_tmp ? "" : X48HOMEDIR,
               opt_gx ? PORT1_GX_FILE : PORT1_SX_FILE );
      if( write_mem_file(fnam, saturn.port1, port1_size) )
         printf( "%s:\tCan't write <%s> port1 file.\n", x48name, fnam );
   }

   /* port2 */
   if( (port2_size > 0) && port2_is_ram )
   {
      sprintf( fnam, "%s%s/%s", in_tmp ? "/tmp" : homedir,
               in_tmp ? "" : X48HOMEDIR,
               opt_gx ? PORT2_GX_FILE : PORT2_SX_FILE );
      if( write_mem_file(fnam, saturn.port2, port2_size) )
         printf( "%s:\tCan't write <%s> port2 file.\n", x48name, fnam );
   }

   return 0;
}


/*********************
 * Write saturn file *
 *********************/
int write_saturn_file( char *fname )
{
   FILE *fp;
   int  i;


   if( opt_verbose )
      printf( "%s:\tWriting <%s> saturn file.\n", x48name, fname );

   if( !(fp = fopen(fname, "w")) )
      return -1;

   write_32( fp, (word_32 *)&saturn.magic );
   for( i = 0; i < 4; i++ )
      write_char( fp, &saturn.version[i] );
   for( i = 0; i < 16; i++ )
      write_8( fp, &saturn.A[i] );
   for( i = 0; i < 16; i++ )
      write_8( fp, &saturn.B[i] );
   for( i = 0; i < 16; i++ )
      write_8( fp, &saturn.C[i] );
   for( i = 0; i < 16; i++ )
      write_8( fp, &saturn.D[i] );
   write_32( fp, &saturn.d[0] );
   write_32( fp, &saturn.d[1] );
   write_8( fp, &saturn.P );
   write_32( fp, &saturn.PC );
   for( i = 0; i < 16; i++ )
      write_8( fp, &saturn.R0[i] );
   for( i = 0; i < 16; i++ )
      write_8( fp, &saturn.R1[i] );
   for( i = 0; i < 16; i++ )
      write_8( fp, &saturn.R2[i] );
   for( i = 0; i < 16; i++ )
      write_8( fp, &saturn.R3[i] );
   for( i = 0; i < 16; i++ )
      write_8( fp, &saturn.R4[i] );
   for( i = 0; i < 4; i++ )
      write_8( fp, &saturn.IN[i] );
   for( i = 0; i < 3; i++ )
      write_8( fp, &saturn.OUT[i] );
   write_8( fp, &saturn.CARRY );
   for( i = 0; i < NR_PSTAT; i++ )
      write_8( fp, &saturn.PSTAT[i] );
   write_8( fp, &saturn.XM );
   write_8( fp, &saturn.SB );
   write_8( fp, &saturn.SR );
   write_8( fp, &saturn.MP );
   write_8( fp, &saturn.hexmode );
   for( i = 0; i < NR_RSTK; i++ )
      write_32( fp, &saturn.rstk[i] );
   write_16( fp, (word_16 *)&saturn.rstkp );
   for( i = 0; i < 9; i++ )
      write_16( fp, (word_16 *)&saturn.keybuf.rows[i] );
   write_8( fp, &saturn.intenable );
   write_8( fp, &saturn.int_pending );
   write_8( fp, &saturn.kbd_ien );
   write_8( fp, &saturn.disp_io );
   write_8( fp, &saturn.contrast_ctrl );
   write_8( fp, &saturn.disp_test );
   write_16( fp, &saturn.crc );
   write_8( fp, &saturn.power_status );
   write_8( fp, &saturn.power_ctrl );
   write_8( fp, &saturn.mode );
   write_8( fp, &saturn.annunc );
   write_8( fp, &saturn.baud );
   write_8( fp, &saturn.card_ctrl );
   write_8( fp, &saturn.card_status );
   write_8( fp, &saturn.io_ctrl );
   write_8( fp, &saturn.rcs );
   write_8( fp, &saturn.tcs );
   write_8( fp, &saturn.rbr );
   write_8( fp, &saturn.tbr );
   write_8( fp, &saturn.sreq );
   write_8( fp, &saturn.ir_ctrl );
   write_8( fp, &saturn.base_off );
   write_8( fp, &saturn.lcr );
   write_8( fp, &saturn.lbr );
   write_8( fp, &saturn.scratch );
   write_8( fp, &saturn.base_nibble );
   write_32( fp, &saturn.disp_addr );
   write_16( fp, &saturn.line_offset );
   write_8( fp, &saturn.line_count );
   write_16( fp, &saturn.unknown );
   write_8( fp, &saturn.t1_ctrl );
   write_8( fp, &saturn.t2_ctrl );
   write_32( fp, &saturn.menu_addr );
   write_8( fp, &saturn.unknown2 );
   write_char( fp, &saturn.timer1 );
   write_32( fp, &saturn.timer2 );
   write_32( fp, &saturn.t1_instr );
   write_32( fp, &saturn.t2_instr );
   write_16( fp, (word_16 *)&saturn.t1_tick );
   write_16( fp, (word_16 *)&saturn.t2_tick );
   write_32( fp, &saturn.i_per_s );
   write_16( fp, &saturn.bank_switch );
   for( i = 0; i < NR_MCTL; i++ )
   {
      write_16( fp, &saturn.mem_cntl[i].unconfigured );
      write_32( fp, &saturn.mem_cntl[i].config[0] );
      write_32( fp, &saturn.mem_cntl[i].config[1] );
   }

   fclose(fp);
   return 0;
}


/******************
 * write_mem_file *
 ******************/
int write_mem_file( char *name, word_4 *mem, int size )
{
   FILE *fp;
   word_8 byte;
   int i, j;

   if( opt_verbose )
      printf( "%s:\tWriting <%s> file.\n", x48name, name );

   if( !(fp = fopen(name, "w")) )
      return -1;

   for (i = 0, j = 0; i < size / 2; i++)
   {
      byte = (mem[j++] & 0x0F);
      byte |= (mem[j++] << 4) & 0xF0;
      if( fwrite(&byte, 1, 1, fp) != 1 )
         break;
   }
   if( i != size /2 )
   {
      fclose( fp );
      return -1;
   }

   fclose( fp );
   return 0;
}


int write_8( FILE *fp, word_8 *var )
{
   unsigned char tmp;

   tmp = *var;
   if( fwrite(&tmp, 1, 1, fp) != 1 )
   {
      if( opt_verbose )
         fprintf( stderr, "%s:\tCan\'t write word_8.\n", x48name );
      return 0;
   }
   return 1;
}


int write_char( FILE *fp, char *var )
{
   char tmp;

   tmp = *var;
   if( fwrite(&tmp, 1, 1, fp) != 1 )
   {
      if( opt_verbose )
         fprintf( stderr, "%s:\tCan\'t write char.\n", x48name );
      return 0;
   }
   return 1;
}


int write_16( FILE *fp, word_16 *var )
{
   unsigned char tmp[2];

   tmp[0] = (*var >> 8) & 0xff;
   tmp[1] = *var & 0xff;
   if (fwrite(&tmp[0], 1, 2, fp) != 2 )
   {
      if( opt_verbose )
         fprintf( stderr, "%s:\tCan\'t write word_16.\n", x48name );
      return 0;
   }
   return 1;
}


int write_32( FILE *fp, word_32 *var )
{
   unsigned char tmp[4];

   tmp[0] = (*var >> 24) & 0xff;
   tmp[1] = (*var >> 16) & 0xff;
   tmp[2] = (*var >> 8) & 0xff;
   tmp[3] = *var & 0xff;
   if( fwrite(&tmp[0], 1, 4, fp) != 4 )
   {
      if( opt_verbose )
         fprintf( stderr, "%s:\tCan\'t write word_32.\n", x48name );
      return 0;
   }
   return 1;
}


int write_u_long( FILE *fp, unsigned long *var )
{
   unsigned char tmp[4];

   tmp[0] = (*var >> 24) & 0xff;
   tmp[1] = (*var >> 16) & 0xff;
   tmp[2] = (*var >> 8) & 0xff;
   tmp[3] = *var & 0xff;
   if( fwrite(&tmp[0], 1, 4, fp) != 4 )
   {
      if( opt_verbose )
         fprintf( stderr, "%s:\tCan\'t write unsigned long.\n", x48name );
      return 0;
   }
   return 1;
}

