/*
 * Program: x48 with GTK support
 * Version: 0.5.0 (GTK)
 * File: lcd.c
 * Description: routines that manipulate Hp48 display
 * History:
 *          Ver              Modified                Date
 *          ===  ================================  ========
 */


#include <config.h>
#include <stdio.h>
#include <unistd.h>
#include <string.h>
#ifdef SUNOS
#include <memory.h>
#endif

#include "global.h"
#include "hp48.h"

/* #define DEBUG_DRAW_NIBBLE 1 */


/****************
 * init_display *
 ****************/
void init_display()
{
   display.on = (int) (saturn.disp_io & 0x8) >> 3;

   display.disp_start = saturn.disp_addr & 0xFFFFE;
   display.offset = saturn.disp_io & 0x7;

   display.lines = saturn.line_count & 0x3F;
   if( !display.lines )
      display.lines = 63;
   if( display.lines < 55 )
      display.lines = 55;

   if( display.offset > 3 )
      display.nibs_per_line = (NIBBLES_PER_ROW+saturn.line_offset+2) & 0xFFF;
   else
      display.nibs_per_line = (NIBBLES_PER_ROW+saturn.line_offset) & 0xFFF;

   display.disp_end = display.disp_start +
  	              ( display.nibs_per_line * (display.lines + 1) );

   display.menu_start = saturn.menu_addr;
   display.menu_end = saturn.menu_addr + 0x110;

   display.contrast = saturn.contrast_ctrl;
   display.contrast |= (saturn.disp_test & 0x1) << 4;

   display.annunc = saturn.annunc;

   memset( display.disp_buf, 0x0, sizeof(display.disp_buf) );
}


/************
 * draw_row *
 ************/
static inline void draw_row( long addr, int row )
{
   int i, v;
   int line_length;

   line_length = NIBBLES_PER_ROW;
   if( (display.offset > 3) && (row <= display.lines) )
      line_length += 2;
   for( i = 0; i < line_length; i++ )
   {
      v = read_nibble( addr + i );
      if( v !=  display.disp_buf[row][i] )
      {
         display.disp_buf[row][i] = v;
         draw_nibble( i, row, v );
      }
   }
}


/******************
 * update_display *
 ******************/
void update_display()
{
   int i, j;
   long addr;
   static int old_offset = -1, old_lines = -1;

   if( display.on )
   {
      addr = display.disp_start;

      if( display.offset != old_offset )
      {
         memset( display.disp_buf, 0x00,
                 (size_t) ((display.lines+1) * NIBS_PER_BUFFER_ROW) );
         old_offset = display.offset;
      }

      if( display.lines != old_lines )
      {
         memset( &display.disp_buf[56][0], 0x00,
	         (size_t) (8 * NIBS_PER_BUFFER_ROW) );
         old_lines = display.lines;
      }

      for( i = 0; i <= display.lines; i++ )
      {
         draw_row( addr, i );
         addr += display.nibs_per_line;
      }

     if( i < DISP_ROWS )
     {
        addr = display.menu_start;
        for( ; i < DISP_ROWS; i++ )
        {
           draw_row( addr, i );
           addr += NIBBLES_PER_ROW;
        }
     }
   }
   else
   {
      memset( &display.disp_buf, 0x00, sizeof(display.disp_buf) );
      for( i = 0; i < 64; i++ )
         for( j = 0; j < NIBBLES_PER_ROW; j++ )
            draw_nibble( j, i, 0x00 );
   }
}


/***********************
 * display_draw_nibble *
 ***********************/
void disp_draw_nibble( word_20 addr, word_4 val )
{
   long offset;
   int x, y;

#ifdef DEBUG_DRAW_NIBBLE
   fprintf(stderr, " Draw Nibble %x at %.5lx\n", val, addr );
#endif

   offset = addr - display.disp_start;
   x = offset % display.nibs_per_line;
   if( x < 0 || x > 35 )
      return;
   if( display.nibs_per_line )
   {
      y = offset / display.nibs_per_line;
      if( y < 0 || y > 63 )
         return;
      if( val != display.disp_buf[y][x] )
      {
         display.disp_buf[y][x] = val;
         draw_nibble( x, y, val );
      }
   }
   else
      for( y = 0; y < display.lines; y++ )
         if( val != display.disp_buf[y][x] )
         {
            display.disp_buf[y][x] = val;
            draw_nibble( x, y, val );
         }
}


/********************
 * menu_draw_nibble *
 ********************/
void menu_draw_nibble( word_20 addr, word_4 val )
{
   long offset;
   int x, y;

#ifdef DEBUG_DRAW_NIBBLE
   fprintf(stderr, " Draw Menu Nibble %x at %.5lx\n", val, addr );
#endif

   offset = addr - display.menu_start;
   x = offset % NIBBLES_PER_ROW;
   y = display.lines + (offset / NIBBLES_PER_ROW) + 1;
   if( val != display.disp_buf[y][x] )
   {
      display.disp_buf[y][x] = val;
      draw_nibble( x, y, val );
   }
}

