/*
 * Program: x48 with GTK support
 * Version: 0.5.0 (GTK)
 * File: options.c
 * Description: parse '~/.x48/x48.cfg' and command line for options
 * History:
 *          Ver              Modified                Date
 *          ===  ================================  ========
 */


#include <config.h>
#include <ctype.h>
#include <stdio.h>
#include <string.h>
#include "global.h"


#define X48RC_MAGIC "# "PROGNAME" resource file"


/* options and their value */
static struct {
                 char *opt;
                 void *var;
                 int  val;
                 char *val1;
                 char *val2;
              } options[15] =
{
   { "Hp48 Model", &opt_gx, 1, "gx", "sx" },
   { "Show Intro", &opt_show_intro, 1, "true", "false" },
   { "Ask for Save Files at Exit", &opt_ask_save, 1, "true", "false" },
   { "Save Files at Exit", &opt_save, 1, "true", "false" },
   { "Save Configuration at Exit", &opt_save_cfg, 1, "true", "false" },
   { "Run Verbosely", &opt_verbose, 1, "true", "false" },
   { "Use Serial", &opt_serial, 1, "true", "false" },
   { "Serial Line", &opt_serial_line, 0, NULL, NULL },
   { "Use Terminal", &opt_terminal, 1, "true", "false" },
   { "Use Debug", &opt_debug, 1, "true", "false" },
   { "Mnemonics", &opt_mnemonics_hp, 1, "hp", "class" },
   { "Port1 Writeable", &opt_port1w, 1, "true", "false" },
   { "Port2 Writeable", &opt_port2w, 1, "true", "false" },
   { "Accurate Time", &opt_time_accurate, 1, "true", "false" },
   { "Timer1 Period", &opt_timer1_period, 0, NULL, NULL },
};

/* Functions */
int  parse_line( char *line );
void split_line( char *line, char *opt, char *val );
int  convert_str_1( char *old, char *new );
int  convert_str_2( char *old, char *new );


/********************************************************
 * parse_x48rc: parse '~/.x48/x48.cfg' file for options *
 ********************************************************/
int parse_x48rc()
{
   FILE *fx48rc;
   char file[255], line[1024], *buf;
   int  ch;

   sprintf( file, "%s%s", homedir, X48RC_FILE );
   if( ! ( fx48rc = fopen(file, "r") ) )
   {
      fprintf( stderr, "%s:\tCan't open <%s>\n", x48name, file );
      fprintf( stderr, "\tTrying default options.\n\n\n" );
      force_save = 1;
      return -1;
   }

   /* check first line of file for header */
   buf = line;
   while( (ch = fgetc(fx48rc)) != EOF )
   {
      if( ch != '\n' )
      {
         *(buf++) = (char) ch;
         continue;
      }
      *buf = '\0';
      if( strcmp( line, X48RC_MAGIC ) )
      {
         fprintf( stderr, "%s:\t<%s> is not a correct resource file,\n",
                  x48name, file );
         fprintf( stderr, "\tThe header of this file must be '%s'\n",
                  X48RC_MAGIC );
         fprintf( stderr, "\tand not '%s'.\n", line );
         return -1;
      }
      break;
   }

   /* next lines */
   buf = line;
   while( (ch = fgetc(fx48rc)) != EOF )
   {
      if( ch != '\n' )
      {
         *(buf++) = (char) ch;
         continue;
      }
      *buf = '\0';

      /* skip comments and blank lines */
      if( line[0] == '#' || line[0] == '\0' )
         continue;

      if( parse_line(line) )
         force_save_cfg = 1;
      buf = line;
   }

   return 0;
}


int parse_line( char *line )
{
   char optn[1024], valn[1024], opt[1024], val[1024], buf[50];
   int  i, num_options;

   num_options = sizeof(options) / sizeof(options[0]);

   split_line( line, optn, valn );
   if( convert_str_1(optn, opt) )
   {
      if( opt_verbose )
         printf( "%s:\tError parsing %s: line not valid '%s'.\n",
                 x48name, X48RC_FILE, optn );
      return -1;
   }

   for( i = 0; i < num_options; i++ )
   {
      convert_str_1( options[i].opt, buf );
      if( !strcmp(opt, buf) )
         break;
   }
   if( i == num_options )
   {
      if( opt_verbose )
         printf( "%s:\tError parsing %s: line does not contain a valid option '%s'.\n",
                 x48name, X48RC_FILE, optn );
      return -1;
   }

   if( options[i].val )
   {
      if( convert_str_1(valn, val) )
      {
         if( opt_verbose )
            printf( "%s:\tError parsing %s: line contains an invalid value for this option '%s'.\n",
                    x48name, X48RC_FILE, valn );
         return -1;
      }
      if( !(strcmp(val, options[i].val1)) )
         *( (int *) options[i].var) = 1;
      else if( !(strcmp(val, options[i].val2)) )
         *( (int *) options[i].var) = 0;
      else
      {
         if( opt_verbose )
            printf( "%s:\Error parsing %s: line contains an invalid value for this option '%s'.\n",
                    x48name, X48RC_FILE, valn );
         return -1;
      }
   }
   else
   {
      if( convert_str_2(valn, val) )
      {
         if( opt_verbose )
            printf( "%s:\tError parsing %s: line contains an invalid value for this option '%s'.\n",
                    x48name, X48RC_FILE, valn );
         return -1;
      }
      strcpy( (char *) options[i].var, val );
   }
   return 0;
}


void split_line( char *line, char *opt, char *val )
{
   char c;

   while( (c=*(line++)) &&  c!='=' )
      *(opt++) = c;
   *opt = '\0';

   if( c )
      while( ( *(val++) = *(line++) ) )
         ;
   else
      *val = '\0';
}


int convert_str_1( char *old, char *new )
{
   char c;

   while( ( c = *(old++) ) )
   {
      if( isspace((int) c) )
         continue;
      if( isalpha((int) c) )
         *(new++) = (char) tolower( (int) c );
      else if( isgraph((int) c) )
         *(new++) = c;
      else
         return -1;
   }
   *new = '\0';
   return 0;
}


int convert_str_2( char *old, char *new )
{
   char c;

   while( ( c = *(old++) ) )
   {
      if( isspace((int) c) )
         continue;
      if( isgraph((int) c) )
         *(new++) = c;
      else
         return -1;
   }
   *new = '\0';
   return 0;
}


/**************************************
 * show_options: show choosen options *
 **************************************/
void show_options()
{
   int i;

   printf( "%s:\tResults of parsing ~/%s\n", x48name, X48RC_FILE );
   for( i = 0; i < sizeof(options) / sizeof(options[0]); i++ )
      if( options[i].val )
         printf( "\t\tOption: %28s\tValue: %s\n", options[i].opt,
                 *((int *) options[i].var) ? options[i].val1 : options[i].val2 );
      else
         printf( "\t\tOption: %28s\tValue: %s\n", options[i].opt,
                 (char *) options[i].var );
   printf( "\n" );
}


/*******************************************************
 * write_x48rc: save '~/.x48/x48.cfg' file for options *
 *******************************************************/
int write_x48rc()
{
   FILE *fx48rc;
   char file[255];
   int  i, num_options;

   if( opt_verbose )
      printf( "%s:\tWriting <%s%s> config file.\n", x48name, homedir,
              X48RC_FILE );

   sprintf( file, "%s%s", homedir, X48RC_FILE );
   if( ! ( fx48rc = fopen(file, "w") ) )
   {
      fprintf( stderr, "%s:\tCan't open <%s> to write in.\n", x48name, file );
      return -1;
   }

   fprintf( fx48rc, "%s\n\n", X48RC_MAGIC );

   num_options = sizeof(options) / sizeof(options[0]);
   for( i = 0; i < num_options; i++ )
   {
      if( options[i].val )
         fprintf( fx48rc, "%s = %s\n", options[i].opt,
               *( int *) options[i].var ? options[i].val1 : options[i].val2 );
      else
         fprintf( fx48rc, "%s = %s\n", options[i].opt, (char *) options[i].var );
   }

   fprintf( fx48rc, "\n############# End of %s #############", file );

   fclose( fx48rc );
   return 0;
}


/******************************************************
 * parse_args: parse command line looking for options *
 ******************************************************/
int parse_args( int argc, char *argv[] )
{
   int i = argc;

   /* Arguments */
   while( --i )
   {
      if( argv[i][0] == '-' )
      {
         if( !strcmp(argv[i], "-verbose") )
         {
            opt_verbose = 1;
            continue;
         }
         if( !strcmp(argv[i], "-quiet") )
         {
            opt_verbose = 0;
            continue;
         }
         if( !strcmp(argv[i], "-initialize") )
         {
            opt_initialize = 1;
            force_save = 1;
            force_save_cfg = 1;
            continue;
         }
         if( !strcmp(argv[i], "-reset") )
         {
            opt_reset = 1;
            continue;
         }
         if( !strcmp(argv[i], "-gx") )
         {
            opt_gx = 1;
            continue;
         }
         if( !strcmp(argv[i], "-sx") )
         {
            opt_gx = 0;
            continue;
         }
         if( !strcmp(argv[i], "-debug") )
         {
            opt_debug = 1;
            continue;
         }
         if( !strcmp(argv[i], "-no-debug") )
         {
            opt_debug = 0;
            continue;
         }
         if( !strcmp(argv[i], "-hp") )
         {
            opt_mnemonics_hp = 1;
            continue;
         }
         if( !strcmp(argv[i], "-class") )
         {
            opt_mnemonics_hp = 0;
            continue;
         }
         if( !strcmp(argv[i], "-help") || !strcmp(argv[i], "-h") )
         {
            usage();
            exit( 0 );
         }
         fprintf( stderr, "Ignoring option %s\n", argv[i] );
      }
      else
         fprintf( stderr, "Ignoring '%s'\n", argv[i] );
   }

   return 0;
}


/***************
 * usage: help *
 ***************/
void usage()
{
   fprintf( stderr, "Usage:\n\t%s\t[-options ...]\n", x48name );
   fprintf( stderr, "\nwhere options include:\n" );
   fprintf( stderr, "    -help / -h   Print this help.\n" );
   fprintf( stderr, "    -verbose     Run verbosive.\n" );
   fprintf( stderr, "    -quiet       Run quietly.\n" );
   fprintf( stderr, "    -initialize  Initialize calculator.\n" );
   fprintf( stderr, "    -reset       Reset calculator at start.\n" );
   fprintf( stderr, "    -gx          Select HP 48 GX model.\n" );
   fprintf( stderr, "    -sx          Select HP 48 SX model.\n" );
   fprintf( stderr, "    -debug       Start with debug mode actived.\n" );
   fprintf( stderr, "    -no-debug    Start without debug mode.\n" );
   fprintf( stderr, "    -hp          Select HP mnemonics.\n" );
   fprintf( stderr, "    -class       Select Class mnemonics.\n" );
   fprintf( stderr,
            "\nOther options can be indicated via the resource file <%s%s>\n",
            homedir, X48RC_FILE );
   fprintf( stderr, "or in the 'setting' dialog inside the program.\n\n" );
}

