/*
 * Program: x48 with GTK support
 * Version: 0.5.0 (GTK)
 * File: rpl.c
 * Description: functions related with RPL, stack, etc.
 * History:
 *          Ver              Modified                Date
 *          ===  ================================  ========
 */


#include <config.h>
#include <stdio.h>
#include <stdlib.h>
#include <sys/stat.h>

#include "global.h"
#include "hp48.h"
#include "rpl.h"


/******************
 * read_nibbles_2 *
 ******************/
long read_nibbles_2( long addr, int len )
{
   long val = 0;

   addr += len;
   while( len-- > 0 )
      val = (val << 4) | ( *(char *) --addr);

   return val;
}


/*********************************************************************
 * RPL_Pick: get the address of the object in stack level 'l'        *
 *********************************************************************/
word_20 RPL_Pick( int l )
{
   word_20 stkp;

   if( !l )
      return 0x00000;
   stkp = read_nibbles( DSKTOP, 5 );
   return read_nibbles( stkp + (l-1)*5, 5 );
}


/*********************************************************************
 * RPL_Push: push the object in n into stack level 1                 *
 *********************************************************************/
void RPL_Push( word_20 n )
{
   word_20 avmem, stkp;

   if( !(avmem = read_nibbles(AVMEM, 5)) )
      return;
   write_nibbles( AVMEM, --avmem, 5 );
   stkp = read_nibbles( DSKTOP, 5 );
   stkp -= 5;
   write_nibbles( stkp, n, 5 );
   write_nibbles( DSKTOP, stkp, 5 );
}


/*********************************************************************
 * RPL_Replace: replace the object in stack level 1 by the one in 'n'*
 *********************************************************************/
void RPL_Replace( word_20 n )
{
   word_20 stkp;

   stkp = read_nibbles( DSKTOP, 5 );
   write_nibbles( stkp, n, 5 );
}


/*********************************************************************
 * RPL_CreateTemp: return address to 'l' free nibs in Temp Memory    *
 *********************************************************************/
word_20 RPL_CreateTemp( word_20 l )
{
   word_20 a, b, c;
   char    *p, *q;
   int     i;

   l += 6;
   a = read_nibbles( TEMPTOP, 5 );
   b = read_nibbles( RSKTOP, 5 );       /* start of avalaible memory */
   c = read_nibbles( DSKTOP, 5 );       /* end of avalaible memory */
   if( (b+l) > c )                      /* check if enough room */
      return 0x00000;

   write_nibbles( TEMPTOP, a+l, 5 );    /* adjust end of temporary objs */
   write_nibbles( RSKTOP, b+l, 5 );     /* adjust start of temporary mem */
   write_nibbles( AVMEM, (c-(b+l))/5, 5 );     /* adjust new free memory */

   p = (char *) malloc( (size_t) (b-a) );
   q = p;
   for( i = 0; i < (b-a); i++ )
      *(q++) = (char) read_nibble( a+i );
   for( i = 0; i < (b-a); i++ )
      write_nibble( a+l+i, (int) *(p++) );
/*   free( p ); */

   write_nibbles( a+l-5, l, 5 );        /* set temporary obj length field */
   return( a+1 );                       /* return temporary obj address */
}


/*********************************************************************
 * RPL_ObjectSize: return the size (in nibbles) of the object in 'd' *
 *********************************************************************/
word_20 RPL_ObjectSize( word_20 d )
{
   word_20 prolog, l = 0, n;

   prolog = read_nibbles( d, 5 );
   switch( prolog )
   {
      case DOBINT   :        l = 10; break;
      case DOREAL   :        l = 21; break;
      case DOEREL   :        l = 26; break;
      case DOCMP    :        l = 37; break;
      case DOECMP   :        l = 69; break;
      case DOCHAR   :        l =  7; break;
      case DOACPTR  :        l = 15; break;
      case DOROMP   :        l = 11; break;

      case DOLIST   :
      case DOSYMB   :
      case DOEXT    :
      case DOCOL    :        n = 5;
                             while( n )
                             {
                                l += n;
                                d += n;
                                n = RPL_ObjectSize( d );
                             };
                             l += 5;
                             break;

      case DOIDNT   :
      case DOLAM    :        l = 7 + read_nibbles( d+5, 2 ) * 2;
                             break;

      case DOTAG    :        n = 7 + read_nibbles( d+5, 2 ) * 2;
                             l = n + RPL_ObjectSize( d+n );
                             break;

      case DORRP    :        if( !(n = read_nibbles(d+8, 5)) )
                                l = 13;
                             else
                             {
                                l = n + 8;
                                n = read_nibbles( d+l, 2 ) * 2 + 4;
                                l += n;
                                l += RPL_ObjectSize( d+l );
                             }
                             break;

      case DOARRY   :
      case DOLNKARRY:
      case DOCSTR   :
      case DOHXS    :
      case DOGROB   :
      case DOLIB    :
      case DOBAK    :
      case DOEXT0   :
      case DOEXT2   :
      case DOEXT3   :
      case DOEXT4   :
      case DOCODE   :        l = 5 + read_nibbles( d+5, 5 );
                             break;

      case SEMI     :        l =  0; break;

      default:               l =  5; break;
   }

   return l;
}


/*********************************************************************
 * RPL_SkipOb: return the address after skiping object in 'd'        *
 *********************************************************************/
word_20 RPL_SkipOb( word_20 d )
{
   return( d + RPL_ObjectSize(d) );
}


/*********************************************************************
 * RPL_ObjectSize_2: same as RPL_ObjectSize but in computer memory   *
 *********************************************************************/
word_20 RPL_ObjectSize_2( word_20 d )
{
   word_20 prolog, l = 0, n;

   prolog = read_nibbles_2( d, 5 );
   switch( prolog )
   {
      case DOBINT   :        l = 10; break;
      case DOREAL   :        l = 21; break;
      case DOEREL   :        l = 26; break;
      case DOCMP    :        l = 37; break;
      case DOECMP   :        l = 69; break;
      case DOCHAR   :        l =  7; break;
      case DOACPTR  :        l = 15; break;
      case DOROMP   :        l = 11; break;

      case DOLIST   :
      case DOSYMB   :
      case DOEXT    :
      case DOCOL    :        n = 5;
                             while( n )
                             {
                                l += n;
                                d += n;
                                n = RPL_ObjectSize_2( d );
                             };
                             l += 5;
                             break;

      case DOIDNT   :
      case DOLAM    :        l = 7 + read_nibbles_2( d+5, 2 ) * 2;
                             break;

      case DOTAG    :        n = 7 + read_nibbles_2( d+5, 2 ) * 2;
                             l = n + RPL_ObjectSize_2( d+n );
                             break;

      case DORRP    :        if( !(n = read_nibbles_2(d+8, 5)) )
                                l = 13;
                             else
                             {
                                l = n + 8;
                                n = read_nibbles_2( d+l, 2 ) * 2 + 4;
                                l += n;
                                l += RPL_ObjectSize_2( d+l );
                             }
                             break;

      case DOARRY   :
      case DOLNKARRY:
      case DOCSTR   :
      case DOHXS    :
      case DOGROB   :
      case DOLIB    :
      case DOBAK    :
      case DOEXT0   :
      case DOEXT2   :
      case DOEXT3   :
      case DOEXT4   :
      case DOCODE   :        l = 5 + read_nibbles_2( d+5, 5 );
                             break;

      case SEMI     :        l =  0; break;

      default:               l =  5; break;
   }

   return l;
}


/*********************************************************************
 * LoadObject: load an HP48 object or an ascii file in temporary mem *
 *********************************************************************/
int LoadObject( char *filename )
{
   FILE        *f;
   struct stat st;
   long        size, i, j;
   word_20     len, addr;
   int         binary, byte;
   char        header[8], *buf;

   if( !(f = fopen(filename, "r")) )
      return( -1 );                     /* ERROR: can't open file */
   if( stat(filename, &st)  < 0 )
   {
      fclose( f );
      return( -2 );                     /* ERROR: can't stat file */
   }
   if( (size = st.st_size) > 131072 )   /* 128 kbs */
   {
      fclose( f );
      return( -3 );                     /* ERROR: file too big */
   }
   if( fread(header, 1, 8, f) != 8 )
   {
      fclose( f );
      return( -4 );                     /* ERROR: can't read file header */
   }

   /* check if HP48 file or not */
   if( memcmp(header, "HPHP48-", 7) )
   {
      binary = 0;
      fseek( f, 0, SEEK_SET );
   }
   else
   {
      binary = 1;
      size -= 8;
   }

   /* save room for file */
   if( !(buf = (char *) malloc(size*2)) )
   {
      fclose( f );
      return( -5 );                     /* ERROR: can't save enough room */
   }
   /* copy file into buffer */
   for( i = 0, j = 0; i < size; i++ )
   {
      if( fread(&byte, 1, 1, f) != 1 )
      {
         fclose( f );
         free( buf );
         return( -6 );                  /* ERROR: can't read from file */
      }
      buf[j++] = (char) byte & 0x0F;
      buf[j++] = (char) (byte >> 4) & 0x0F;
   }
   fclose( f );

   /* introduce file into HP48 memory */
   if( binary )         /* BINARY file */
   {
      len = RPL_ObjectSize_2( (word_20) buf );
      if( !(addr = RPL_CreateTemp(len)) )
      {
         fprintf( stderr, "Object address = 0x%5lX\n", len );
         free( buf );
         return( -7 );                  /* ERROR: HP48 has not enough memory */
      }
      for( i = 0; i < len; i++ )
         write_nibble( addr+i, (int) buf[i] );
   }

   else                 /* ASCII file */
   {
      len = size * 2;
      if( !(addr = RPL_CreateTemp(len+10)) )
      {
         free( buf );
         return( -7 );                  /* ERROR: HP48 has not enough memory */
      }
      write_nibbles( addr, 0x02A2C, 5 );
      write_nibbles( addr+5, len+5, 5 );
      for( i = 0; i < len; i++ )
         write_nibble( addr+10+i, (int) buf[i] );
   }

   RPL_Push( addr );

   free( buf );
   return 0;
}


/*********************************************************************
 * SaveObject: save an HP48 object in a file with 'filename' name    *
 *********************************************************************/
int SaveObject( word_20 obj, char *filename )
{
   FILE    *f;
   word_20 len;
   int     byte, add_1_nib = 0;

   len = RPL_ObjectSize( obj );         /* size in nibbles */
   if( len & 1 )
      add_1_nib = 1;
   len /= 2;                            /* size in bytes */

   if( !(f = fopen( filename, "w" )) )
      return( -1 );                     /* ERROR: can't open file */

   fprintf( f, "HPHP48-X" );
   while( len-- )
   {
      byte =  read_nibble(obj++) | (read_nibble(obj++) << 4);
      if( fwrite( &byte, 1, 1, f) != 1 )
      {
         fclose( f );
         return( -2 );                  /* ERROR: can't write in file */
      }
   }
   if( add_1_nib )
   {
      byte = read_nibble( obj );
      fwrite( &byte, 1, 1, f );
   }
   fclose( f );
   return 0;
}

